<?php
require_once '../includes/config.php';
require_once '../includes/Database.php';
require_once '../includes/functions.php';
require_once '../includes/phpmailer-email-config.php';

// Only allow POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    redirect_to('../');
}

// Detect which page request came from
$pageFrom = trim($_POST['page_from'] ?? 'index');
if($pageFrom === 'contact') {
    $redirectPage = '../contact';
} elseif($pageFrom === 'founder-details') {
    $redirectPage = '../founder-details?id=' . (int)($_POST['founder-details-id'] ?? 0);
} else {
    $redirectPage = '../';
}

// verify form submission
if (!isset($_POST['contact-form-submit'])) {
    redirect_to($redirectPage);
}

// Get inputs
$name    = trim($_POST['name'] ?? '');
$email   = trim($_POST['email'] ?? '');
$subject = trim($_POST['subject'] ?? '');
$message = trim($_POST['message'] ?? '');

// Save previous session values
function save_old_session_inputs($name, $email, $subject, $message) {
    set_flash_message("contact_old_session_name", $name);
    set_flash_message("contact_old_session_email", $email);
    set_flash_message("contact_old_session_subject", $subject);
    set_flash_message("contact_old_session_message", $message);
}


// Validation: required fields
if ($name === '' || $email === '' || $subject === '' || $message === '') {
    set_flash_message("contact_message", [
        "type" => "error",
        "text" => "All required fields must be filled."
    ]);

    save_old_session_inputs($name, $email, $subject, $message);

    redirect_to($redirectPage);
}

// Email validation
if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
    set_flash_message("contact_message", [
        "type" => "error",
        "text" => "Please enter a valid email address."
    ]);

    save_old_session_inputs($name, $email, $subject, $message);

    redirect_to($redirectPage);
}

// Message max length check
if (strlen($message) > 5000) {
    set_flash_message("contact_message", [
        "type" => "error",
        "text" => "Message/Comment must be less than 5000 characters."
    ]);

    save_old_session_inputs($name, $email, $subject, $message);

    redirect_to($redirectPage);
}

// Insert into DB
$db = new Database();

$save = $db->execute(
    "INSERT INTO contacts (name, email, subject, message, created_at) 
     VALUES (?, ?, ?, ?, NOW())",
    [$name, $email, $subject, $message]
);

// Success / Failure response
if ($save) {
    // Send mail Start
    $emailBody = phpmailer_email_template(
        "New Contact Message",
        "
            <p><strong style='color:#224abe;'>Name:</strong> " . escape_html($name) . "</p>
            <p><strong style='color:#224abe;'>Email:</strong> " . escape_html($email) . "</p>
            <p><strong style='color:#224abe;'>Subject:</strong> " . escape_html($subject) . "</p>

            <div style='margin-top:20px;padding:15px;background:#f1f3f9;
                        border-left:4px solid #4e73df;border-radius:6px;'>
                <strong style='color:#224abe;'>Message:</strong><br>
                " . nl2br(escape_html($message)) . "
            </div>
        "
    );
    send_email_with_phpmailer(
        "support@yourdomain.com",          // To Email
        "Admin",                           // To Name
        "New Contact Message - $subject",  // Subject
        $emailBody,                        // HTML Body
        "no-reply@yourdomain.com",         // From Email
        "Your Website"                     // From Name
    );
    // Send mail End

    set_flash_message("contact_message", [
        "type" => "success",
        "text" => "Thank you! Your message has been sent successfully."
    ]);
} else {
    set_flash_message("contact_message", [
        "type" => "error",
        "text" => "Something went wrong. Please try again."
    ]);

    save_old_session_inputs($name, $email, $subject, $message);
}

// Redirect back to original page
redirect_to($redirectPage);
